-- This Source Code Form is subject to the terms of the bCDDL, v. 1.1.
-- If a copy of the bCDDL was not distributed with this
-- file, You can obtain one at http://beamng.com/bCDDL-1.1.txt

local M = {}
local SmoothedTable = {}

-- Read the current handbrake input safely
local function safeReadHandbrake()
  return electrics.values.parkingbrake
      or electrics.values.parkingbrake_input
      or electrics.values.handbrake
      or 0
end

-- Add smoothed, inverted handbrake trigger
local function addHandbrakeTrigger(outputName, smoothSpeed)
  local st = {}
  st.type = "smoothTrigger"
  st.electric = outputName or "parkingbrake_trigger"
  st.speed = smoothSpeed or 2.0  -- lower = slower transition, higher = faster
  electrics.values[st.electric] = 1 -- start as "off" inverted
  table.insert(SmoothedTable, st)
end

-- Update with smoothing
local function updateSmoothTrigger(st, dt)
  local hb = safeReadHandbrake()
  local target = (hb > 0.1) and 0 or 1 -- inverted: engaged → 0, released → 1
  local current = electrics.values[st.electric] or 0
  local rate = st.speed

  -- move current towards target at rate units/sec
  if current < target then
    current = math.min(current + dt * rate, target)
  elseif current > target then
    current = math.max(current - dt * rate, target)
  end

  electrics.values[st.electric] = current
end

local function init()
  SmoothedTable = {}
  -- Creates a smoothed inverted trigger named "parkingbrake_trigger"
  -- Speed = 3 means it will take ~0.33s to go from 0 to 1
  addHandbrakeTrigger("parkingbrake_trigger", 3.0)
end

local function updateGFX(dt)
  for _, st in ipairs(SmoothedTable) do
    if st.type == "smoothTrigger" then
      updateSmoothTrigger(st, dt)
    end
  end
end

M.onInit = init
M.onReset = init
M.updateGFX = updateGFX

return M
