-- ORI dual-chamber air-strut controller (phenomenological)
-- Simulates two nitrogen chambers (top/rod-side and bottom/piston-side)
-- with polytropic gas law, simple bleed equalization, and optional
-- hydraulic bump/top-out forces. Forces apply along the specified shock
-- beam axis, coexisting with a separate damping controller.

local M = {}
M.type = "auxiliary"
M.defaultOrder = 79 -- run before bypassDampers (80) so damping sees final state if needed

local abs = math.abs
local min, max = math.min, math.max

local function clamp(x, a, b)
  if x < a then return a end
  if x > b then return b end
  return x
end

local function lerp(a, b, t)
  return a + (b - a) * t
end

local defaultCfg = {
  n = 1.25,                 -- polytropic exponent (1.0..1.4)
  bleedCoef = 2.0,          -- 1/s simple equalization rate between chambers
  -- advanced bleed shaping (direction/position/velocity sensitive)
  advancedBleed = true,
  bleedCompMul = 1.2,       -- multiplier on compression
  bleedRebMul  = 0.9,       -- multiplier on rebound (often smaller for platform)
  bleedPosCenter = 0.5,     -- normalized center of platform/flow focus
  bleedPosWidth  = 0.25,    -- half-width window (0..0.5); 0.25 -> wide mid band
  bleedPosMin    = 0.4,     -- min factor outside platform band
  bleedPosMax    = 1.4,     -- max factor at center
  bleedVelGain   = 0.8,     -- additional flow with shaft speed
  bleedVelRef    = 0.6,     -- m/s to add ~bleedVelGain
  bleedVelMax    = 2.0,     -- clamp multiplier contribution
  pistonDiameterMm = 76.2,  -- mm, default piston (front-like)
  rodDiameterMm = 22.0,     -- mm, default rod
  topPsi = 60.0,            -- psi initial top (rod-side) pressure
  botPsi = 40.0,            -- psi initial bottom (piston-side) pressure
  -- optional scaling (e.g., if bottom chamber is represented by two PRESSURED beams in JBEAM,
  -- set botPsiScale = 0.5 so the slider shows full chamber PSI but each beam gets half)
  topPsiScale = 1.0,
  botPsiScale = 1.0,
  topVolumeCc = 450.0,      -- cc initial top gas volume at rest
  botVolumeCc = 650.0,      -- cc initial bottom gas volume at rest
  strokeM = 0.25,           -- m nominal stroke for end-zone fractions
  -- velocity-dependent polytropic behavior (blend toward adiabatic with speed)
  nVelRef = 0.8,            -- m/s where we approach configured n; below this tends to 1.0 (isothermal)
  -- End-zone hydraulic bump/top-out (force-only, not damping)
  bumpStartFrac = 0.85,
  bumpK = 20000.0,          -- N at full compression with exp shaping
  bumpExp = 2.2,
  topoutStartFrac = 0.85,
  topoutK = 12000.0,
  topoutExp = 2.0
}

local dampers = {}

-- Convert psi and cc to SI pre-factors for P*V^n = const
local function initChamberConstants(d)
  local PaPerPsi = 6894.757
  local Vt0 = (d.topVolumeCc or defaultCfg.topVolumeCc) * 1e-6    -- m^3
  local Vb0 = (d.botVolumeCc or defaultCfg.botVolumeCc) * 1e-6    -- m^3
  Vt0 = max(1e-8, Vt0)
  Vb0 = max(1e-8, Vb0)
  d._Vt0 = Vt0
  d._Vb0 = Vb0
  local n = d.n or defaultCfg.n
  d.n = n
  local topScale = d.topPsiScale or defaultCfg.topPsiScale or 1.0
  local botScale = d.botPsiScale or defaultCfg.botPsiScale or 1.0
  local Pt0 = ((d.topPsi or defaultCfg.topPsi) * topScale) * PaPerPsi
  local Pb0 = ((d.botPsi or defaultCfg.botPsi) * botScale) * PaPerPsi
  -- store polytropic constants; we will adjust toward equalization via simple bleed
  d._Kt = Pt0 * (Vt0 ^ n)
  d._Kb = Pb0 * (Vb0 ^ n)
  d._Pt = Pt0
  d._Pb = Pb0
end

local function areaFromDiameterMm(mm)
  local m = max(0, (mm or 0) * 0.001)
  return math.pi * 0.25 * m * m
end

local function updateDamper(d, dt)
  if not d.beamCid then return end
  -- current displacement (m), positive on compression
  local L = obj:getBeamLength(d.beamCid) or d.initialBeamLength or 0
  local x = (d.initialBeamLength or L) - L
  if not d.initialized then
    d.prevX = x
    d.initialized = true
  end
  d.prevX = d.prevX or x
  local vel = (x - d.prevX) / max(dt, 1e-6) -- avoid shadowing global BeamNG 'v'
  d.prevX = x

  -- Geometry
  local Ap = d._Ap      -- piston area (m^2)
  local Ar = d._Ar      -- rod area (m^2)
  -- Effective polytropic exponent: near-isothermal at tiny speeds, approach configured n with speed
  local nBase = d.n or defaultCfg.n
  local nRef  = d.nVelRef or defaultCfg.nVelRef or 1.0
  local nBlend = 1.0
  if nRef and nRef > 0 then
    local s = math.min(1, math.abs(vel or 0) / nRef)
    nBlend = (s*s) * (3 - 2*s) -- smoothstep
  end
  local n  = 1.0 + (nBase - 1.0) * nBlend

  -- Chamber volumes under displacement x
  local Vt = max(1e-8, d._Vt0 - Ar * x) -- top shrinks as rod enters
  local Vb = max(1e-8, d._Vb0 + Ap * x) -- bottom expands on compression

  -- Ideal polytropic pressures from stored constants
  local Pt_raw = d._Kt / (Vt ^ n)
  local Pb_raw = d._Kb / (Vb ^ n)

  -- Simple bleed equalization: smoothly reduce dP over time
  local dP = Pb_raw - Pt_raw
  -- Direction sensitivity (compression vs rebound)
  local dirMul = (vel or 0) >= 0 and (d.bleedCompMul or defaultCfg.bleedCompMul or 1.0)
                                or (d.bleedRebMul  or defaultCfg.bleedRebMul  or 1.0)
  -- Position window around mid-stroke to emulate platform valve focus
  local stroke = d.strokeM or defaultCfg.strokeM or 0.25
  local pn = (stroke > 0) and clamp((x / stroke + 0.5), 0, 1) or 0.5  -- map [-stroke..+stroke] roughly to [0..1]
  local c  = d.bleedPosCenter or defaultCfg.bleedPosCenter or 0.5
  local w  = d.bleedPosWidth  or defaultCfg.bleedPosWidth  or 0.25
  local posT = 0
  if w and w > 1e-6 then
    local dist = math.abs(pn - c) / w
    posT = clamp(1 - dist, 0, 1) -- 1 at center, 0 outside window
  end
  local posMul = (d.bleedPosMin or defaultCfg.bleedPosMin or 0.5)
  posMul = posMul + ( (d.bleedPosMax or defaultCfg.bleedPosMax or 1.5) - posMul ) * posT
  -- Velocity amplification of transfer
  local vg = d.bleedVelGain or defaultCfg.bleedVelGain or 0
  local vref = d.bleedVelRef or defaultCfg.bleedVelRef or 1
  local vmax = d.bleedVelMax or defaultCfg.bleedVelMax or 2
  local velMul = 1 + math.min(vmax, (vg or 0) * math.abs(vel or 0) / math.max(vref, 1e-6))
  local kBase = math.max(0, d.bleedCoef or defaultCfg.bleedCoef or 0)
  local a = clamp(kBase * dirMul * posMul * velMul * (dt or 0), 0, 0.7)
  local Pt = Pt_raw + a * dP
  local Pb = Pb_raw - a * dP
  d._Pt, d._Pb = Pt, Pb

  -- Net gas force along axis (bottom pushes out, top pulls in)
  local Fgas = Pb * Ap - Pt * Ar

  -- Mid-stroke platform shaping (optional small centering spring in a window)
  if d.platformK and d.platformK ~= 0 then
    local pc = d.platformCenterFrac or 0.5
    local pw = d.platformWidthFrac or 0.2
    local pn2 = (stroke > 0) and clamp((x / stroke + 0.5), 0, 1) or 0.5
    local t = 0
    if pw and pw > 1e-6 then
      local dist = (pn2 - pc)
      local absn = math.abs(dist) / pw
      if absn < 1 then
        local wlin = 1 - absn
        -- Centering force toward pc (negative when pn2>pc so it pulls back)
        t = - (dist / math.max(pw, 1e-6)) * wlin
      end
    end
    Fgas = Fgas + (d.platformK or 0) * t
  end

  -- End-zone hydraulic bump/top-out forces (position-shaped)
  local stroke = d.strokeM or defaultCfg.strokeM
  if stroke and stroke > 0 then
    local compFrac = clamp(x / stroke, 0, 1)
    if d.bumpStartFrac and compFrac > d.bumpStartFrac then
      local t = (compFrac - d.bumpStartFrac) / max(1e-6, 1 - d.bumpStartFrac)
      local Fb = (d.bumpK or defaultCfg.bumpK) * (t ^ (d.bumpExp or defaultCfg.bumpExp))
      Fgas = Fgas + Fb
    end
    local extFrac = clamp((-x) / stroke, 0, 1)
    if d.topoutStartFrac and extFrac > d.topoutStartFrac then
      local t = (extFrac - d.topoutStartFrac) / max(1e-6, 1 - d.topoutStartFrac)
      local Ft = (d.topoutK or defaultCfg.topoutK) * (t ^ (d.topoutExp or defaultCfg.topoutExp))
      Fgas = Fgas - Ft
    end
  end

  -- Apply axial force equally/oppositely to the beam nodes
  local b = v.data.beams[d.beamCid]
  if not b then return end
  local i1, i2 = b.id1, b.id2
  if not (i1 and i2) then return end
  local x1,y1,z1 = obj:getNodePosition(i1)
  local x2,y2,z2 = obj:getNodePosition(i2)
  if not (x1 and y1 and z1 and x2 and y2 and z2) then return end
  local ax, ay, az = (x2 - x1), (y2 - y1), (z2 - z1)
  local len = math.sqrt(ax*ax + ay*ay + az*az)
  if len < 1e-6 then return end
  ax, ay, az = ax/len, ay/len, az/len
  obj:applyForce(i1, -ax*Fgas, -ay*Fgas, -az*Fgas)
  obj:applyForce(i2,  ax*Fgas,  ay*Fgas,  az*Fgas)
end

local function init(jbeamData)
  dampers = {}

  -- parse damper table
  local damperRows = tableFromHeaderTable(jbeamData.dampers or {})
  if #damperRows == 0 then return end

  -- Build name->cid map
  local nameToCid = {}
  for _, b in pairs(v.data.beams or {}) do
    if b.name then nameToCid[b.name] = b.cid end
  end

  for _, row in ipairs(damperRows) do
    local beamName = row.beamName or row[2]
    local cid = nameToCid[beamName]
    if cid then
      local d = {
        name = row.name or beamName,
        beamCid = cid,
        initialBeamLength = obj:getBeamRestLength(cid),
        -- geometry
        pistonDiameterMm = row.pistonDiameterMm or jbeamData.pistonDiameterMm or defaultCfg.pistonDiameterMm,
        rodDiameterMm    = row.rodDiameterMm    or jbeamData.rodDiameterMm    or defaultCfg.rodDiameterMm,
        -- gas params
        n = row.n or jbeamData.n or defaultCfg.n,
        bleedCoef = row.bleedCoef or jbeamData.bleedCoef or defaultCfg.bleedCoef,
        topPsi = row.topPsi or jbeamData.topPsi or defaultCfg.topPsi,
        botPsi = row.botPsi or jbeamData.botPsi or defaultCfg.botPsi,
        topPsiScale = row.topPsiScale or jbeamData.topPsiScale or defaultCfg.topPsiScale,
        botPsiScale = row.botPsiScale or jbeamData.botPsiScale or defaultCfg.botPsiScale,
        topVolumeCc = row.topVolumeCc or jbeamData.topVolumeCc or defaultCfg.topVolumeCc,
        botVolumeCc = row.botVolumeCc or jbeamData.botVolumeCc or defaultCfg.botVolumeCc,
        strokeM = row.strokeM or jbeamData.strokeM or defaultCfg.strokeM,
        -- end zones
        bumpStartFrac = row.bumpStartFrac or jbeamData.bumpStartFrac or defaultCfg.bumpStartFrac,
        bumpK = row.bumpK or jbeamData.bumpK or defaultCfg.bumpK,
        bumpExp = row.bumpExp or jbeamData.bumpExp or defaultCfg.bumpExp,
        topoutStartFrac = row.topoutStartFrac or jbeamData.topoutStartFrac or defaultCfg.topoutStartFrac,
        topoutK = row.topoutK or jbeamData.topoutK or defaultCfg.topoutK,
        topoutExp = row.topoutExp or jbeamData.topoutExp or defaultCfg.topoutExp
      }
      d._Ap = areaFromDiameterMm(d.pistonDiameterMm)
      d._Ar = areaFromDiameterMm(d.rodDiameterMm)
      initChamberConstants(d)
      table.insert(dampers, d)
    else
      log("E", "ORIairshocks.init", "Unknown beamName '" .. tostring(beamName) .. "' for damper '" .. tostring(row.name or "?") .. "'")
    end
  end
end

local function reset(jbeamData)
  for _, d in ipairs(dampers) do
    d.initialized = false
    initChamberConstants(d)
  end
end

local function update(dt)
  for _, d in ipairs(dampers) do
    updateDamper(d, dt)
  end
end

M.init = init
M.reset = reset
M.update = update

return M
