local M = {}

M.outputPorts = {[1] = true, [2] = true}
M.deviceCategories = {clutchlike = true, clutch = true, shaft = true, differential = true}
M.requiredExternalInertiaOutputs = {1, 2}

local primaryOutputID = 1
local secondaryOutputID = 2

local max = math.max
local min = math.min
local abs = math.abs
local sqrt = math.sqrt
local clamp = clamp

local function updateGFX(device, dt)
end

local function parallelUpdateVelocity(device, dt)
  device.inputAV = device.parent[device.parentOutputAVName]
  device[device.primaryOutputAVName] = 0
end

local function seriesUpdateVelocity(device, dt)
  device.inputAV = device.parent[device.parentOutputAVName]
  device[device.primaryOutputAVName] = device.inputAV / device.gearRatio
end

local function parallelUpdateTorque(device, dt)
  device[device.primaryOutputTorqueName] = 0

  local avDiff = device.inputAV - device[device.secondaryOutputAVName] * device.gearRatio
  local lockDampAV = avDiff

  local maxClutchAngle = device.maxClutchAngle
  local clutchFreePlay = device.clutchFreePlay
  local clutchAngle = clamp(device.clutchAngle + avDiff * dt, -maxClutchAngle - clutchFreePlay, maxClutchAngle + clutchFreePlay)

  clutchAngle = device.clutchAngleSmoother:get(clutchAngle)

  local absFreeClutchAngle = max(abs(clutchAngle) - clutchFreePlay, 0)
  local lockTorque = device.lockTorque

  lockDampAV = device.lockDampAVSmoother:get(lockDampAV)

  local torqueDiff = (clamp(min(1, absFreeClutchAngle) * absFreeClutchAngle * sign(clutchAngle) * device.lockSpring + lockDampAV * device.lockDamp, -lockTorque, lockTorque))

  device.clutchAngle = clutchAngle
  device.torqueDiff = torqueDiff or 0
  device[device.secondaryOutputTorqueName] = torqueDiff * device.gearRatio
end

local function seriesUpdateTorque(device)
  device[device.primaryOutputTorqueName] = device.parent[device.parentOutputTorqueName] * device.gearRatio
  device[device.secondaryOutputTorqueName] = 0
  device.torqueDiff = device.children[primaryOutputID].torqueDiff / device.gearRatio or 0
end

local function selectUpdates(device)
  if device.mode == "SERIES" then
    device.gearRatio = device.gearRatio1
    device.velocityUpdate = seriesUpdateVelocity
    device.torqueUpdate = seriesUpdateTorque
  elseif device.mode == "PARALLEL" then
    device.gearRatio = device.gearRatio2
    device.velocityUpdate = parallelUpdateVelocity
    device.torqueUpdate = parallelUpdateTorque
  end
end

local function setMode(device, mode)
  device.mode = mode
  selectUpdates(device)
  device:calculateInertia()
end

local function validate(device)
  if not device.parent.deviceCategories.engine then
    log("E", "iDM_Transmission.validate", "Parent device is not an engine device...")
    log("E", "iDM_Transmission.validate", "Actual parent:")
    log("E", "iDM_Transmission.validate", powertrain.dumpsDeviceData(device.parent))
    return false
  end

  device.lockTorque = device.lockTorque or (device.parent.torqueData.maxTorque * 1.25 + device.parent.maxRPM * device.parent.inertia * math.pi / 30)
  return true
end

local function calculateInertia(device)
  local outputInertia = 0
  local cumulativeGearRatio = 1
  local maxCumulativeGearRatio = 1
  if device.mode == "PARALLEL" then
    if device.children then
      if device.children[secondaryOutputID] then
        outputInertia = device.children[secondaryOutputID].cumulativeInertia
        cumulativeGearRatio = device.children[secondaryOutputID].cumulativeGearRatio
        maxCumulativeGearRatio = device.children[secondaryOutputID].maxCumulativeGearRatio
      end
    end

    device.cumulativeInertia = outputInertia / device.gearRatio / device.gearRatio
    device.cumulativeGearRatio = cumulativeGearRatio * device.gearRatio
    device.maxCumulativeGearRatio = maxCumulativeGearRatio * device.gearRatio
    
    device.cumulativeInertia = min(outputInertia, device.parent.inertia * 0.5)
    device.lockSpring = device.lockSpringBase or (powertrain.stabilityCoef * powertrain.stabilityCoef * device.cumulativeInertia * device.lockSpringCoef) --Nm/rad
    
    device.lockDamp = device.lockDampRatio * sqrt(device.lockSpring * device.cumulativeInertia)
    
    --^2 spring but linear spring after 1 rad
    device.maxClutchAngle = sqrt(device.lockTorque / device.lockSpring) + max(device.lockTorque / device.lockSpring - 1, 0)
    --linear spring
    --device.maxClutchAngle = device.lockTorque / device.lockSpring
    
    device.cumulativeGearRatio = cumulativeGearRatio
    device.maxCumulativeGearRatio = maxCumulativeGearRatio
  elseif device.mode == "SERIES" then
    if device.children then
      if device.children[primaryOutputID] then
        outputInertia = device.children[primaryOutputID].cumulativeInertia
        cumulativeGearRatio = device.children[primaryOutputID].cumulativeGearRatio
        maxCumulativeGearRatio = device.children[primaryOutputID].maxCumulativeGearRatio
      end
    end
      
    device.cumulativeInertia = outputInertia / device.gearRatio / device.gearRatio
    device.cumulativeGearRatio = cumulativeGearRatio * device.gearRatio
    device.maxCumulativeGearRatio = maxCumulativeGearRatio * device.gearRatio
  end
end

local function reset(device, jbeamData)
  device.mode = "SERIES"
  device.gearRatio = 1
  device.cumulativeInertia = 1
  device.cumulativeGearRatio = 1
  device.maxCumulativeGearRatio = 1

  device.outputAV1 = 0
  device.outputAV2 = 0
  device.inputAV = 0
  device.outputTorque1 = 0
  device.outputTorque2 = 0
  device.clutchAngle = 0
  device.torqueDiff = 0
  device.lockDampAVSmoother:reset()
  device.clutchAngleSmoother:reset()

  selectUpdates(device)
end

local function new(jbeamData)
  local device = {
    deviceCategories = shallowcopy(M.deviceCategories),
    requiredExternalInertiaOutputs = shallowcopy(M.requiredExternalInertiaOutputs),
    outputPorts = shallowcopy(M.outputPorts),
    mode = "SERIES",
    name = jbeamData.name,
    type = jbeamData.type,
    inputName = jbeamData.inputName,
    inputIndex = jbeamData.inputIndex,
    gearRatio = 1,
    gearRatio1 = jbeamData.gearRatio1 or 1,
    gearRatio2 = jbeamData.gearRatio2 or 0.5,
    additionalEngineInertia = jbeamData.additionalEngineInertia or 0,
    cumulativeInertia = 1,
    cumulativeGearRatio = 1,
    maxCumulativeGearRatio = 1,
    outputAV1 = 0,
    outputAV2 = 0,
    inputAV = 0,
    outputTorque1 = 0,
    outputTorque2 = 0,
    clutchAngle = 0,
    torqueDiff = 0,
    lockDampRatio = jbeamData.lockDampRatio or 0.15, --1 is critically damped

    lockDampAVSmoother = newExponentialSmoothing(jbeamData.lockDampSmoothing or 0),
    clutchAngleSmoother = newExponentialSmoothing(jbeamData.clutchAngleSmoothing or 0),

    clutchFreePlay = jbeamData.clutchFreePlay or 0.125,
    lockSpringCoef = jbeamData.lockSpringCoef or 1,
    lockTorque = jbeamData.lockTorque,
    lockSpringBase = jbeamData.lockSpring,
    reset = reset,
    validate = validate,
    calculateInertia = calculateInertia,
    setMode = setMode,
    updateGFX = updateGFX,
  }

  primaryOutputID = min(max(jbeamData.primaryOutputID or 1, 1), 2) --must be either 1 or 2
  secondaryOutputID = math.abs(primaryOutputID * 3 - 5) --converts 1 -> 2 and 2 -> 1

  device.primaryOutputTorqueName = "outputTorque" .. tostring(primaryOutputID)
  device.primaryOutputAVName = "outputAV" .. tostring(primaryOutputID)
  device.secondaryOutputTorqueName = "outputTorque" .. tostring(secondaryOutputID)
  device.secondaryOutputAVName = "outputAV" .. tostring(secondaryOutputID)

  selectUpdates(device)

  device.visualType = "cvtGearbox"

  return device
end

M.new = new

return M
