local M = {}
M.type = "auxiliary"
M.relevantDevice = "engine"

local abs = math.abs
local min = math.min
local max = math.max
local floor = math.floor
local sqrt = math.sqrt
local rpmToAV = 0.104719755
local kphToMs = 1 / 3.6

local engine = nil
local iDM_Transmission = nil
local p1_Motors = nil
local p3_Motors = nil
local motors = nil
local throttleSet = nil
local batteryLow = 0.15
local batteryCriticalLow = 0.05
local engineTempHigh = 100
local engineTempReady = 85
local engineTempLow = 80
local freeRPM = 5500
local idleRPM = 1500
local parallelRPM = 1200
local freeAV = freeRPM * rpmToAV
local idleAV = idleRPM * rpmToAV
local parallelAV = parallelRPM * rpmToAV
local idleSpeed = 15 * kphToMs
local idlingTime = 0
local idlingBreak = 0
local glideCoef = 0
local lastBatteryLevel = 0

local unsmoothedCurrentPower = 0
local currentPowerSmoother = newExponentialSmoothing(100)
local engineCurrentPowerSmoother = newExponentialSmoothing(100)
local currentPowerCoefSmoother = newExponentialSmoothing(10000)

local function setMode(mode)
  electrics.values.hevState = mode
  if iDM_Transmission.mode ~= mode then
    currentPowerCoefSmoother:reset()
    iDM_Transmission:setMode(mode)
  end
end

local function updateMotorIgnition()
  if electrics.values.ignitionLevel >= 2 then
    for _, motor in ipairs(p1_Motors) do
      motor.ignitionCoef = 1
    end
    for _, motor in ipairs(p3_Motors) do
      motor.ignitionCoef = 1
    end
    for _, motor in ipairs(motors) do
      motor.ignitionCoef = 1
    end
  elseif electrics.values.ignitionLevel < 2 then
    for _, motor in ipairs(p1_Motors) do
      motor.ignitionCoef = 0
    end
    for _, motor in ipairs(p3_Motors) do
      motor.ignitionCoef = 0
    end
    for _, motor in ipairs(motors) do
      motor.ignitionCoef = 0
    end
  end
end

local function updateEHSState(dt)
  local vehicleController = require("controller/vehicleController/vehicleController")

  local aggression = clamp((vehicleController.drivingAggression - 0.2) / 0.8, 0, 1)
  local aggressionCoef = aggression * aggression
  if input.throttle == 0 and (electrics.values.throttleOverride == nil or electrics.values.throttleOverride == 0) then
    if electrics.values.wheelspeed > 2 then
      glideCoef = glideCoef + (1 - aggressionCoef) * dt
    end
  else
    glideCoef = 0
  end

  local enginePower = engine.outputTorque1 * engine.outputAV1
  
  local currentPower = 0
  for _, motor in ipairs(p3_Motors) do
    currentPower = currentPower + (motor.outputTorque1 * motor.outputAV1)
  end
  for _, motor in ipairs(motors) do
    currentPower = currentPower + (motor.outputTorque1 * motor.outputAV1)
  end
  if #p3_Motors == 0 and iDM_Transmission.mode == "PARALLEL" then
    currentPower = currentPower + (iDM_Transmission.outputTorque2 * iDM_Transmission.outputAV2 * 0.9)
  end
  unsmoothedCurrentPower = currentPower
  currentPower = currentPowerSmoother:get(currentPower)
  local currentPowerCoef = currentPowerCoefSmoother:get(currentPower)
  currentPowerCoef = (currentPowerCoef * aggressionCoef + currentPower * (1 - aggressionCoef)) / 1000

  electrics.values.checkRE = engine.isDisabled and 1 or 0

  local function startEngine()
    if electrics.values.reRunning == 0 and electrics.values.fuel < electrics.values.hybridSOC / 100 * 0.95 then
      if engine.outputAV1 < idleAV * 0.8 then
        engine:deactivateStarter(engine)
        electrics.values.p1RegenThrottle = 0
        electrics.values.p1Throttle = 0.5
      else
        engine:deactivateStarter(engine)
        engine.ignitionCoef = 1
        electrics.values.reRunning = 1
        electrics.values.engineTrottle = throttleSet
        electrics.values.p1Throttle = 0
      end
    end
  end

  local function stopEngine()
    engine:deactivateStarter(engine)
    electrics.values.engineTrottle = 0
    engine.ignitionCoef = 0
    electrics.values.reRunning = 0
    electrics.values.p1Throttle = 0
    electrics.values.p1RegenThrottle = 0
  end

  local function idlingTempKeeper()
    if engine.thermals.coolantTemperature >= engineTempReady then
      idlingBreak = 1
    end
    if idlingBreak == 1 and engine.thermals.coolantTemperature < engineTempLow then
      idlingBreak = 0
    end
  end

  if electrics.values.checkRE == 1 then
    electrics.values.hybridMode = "EV"
  end
  if electrics.values.ignitionLevel == 2 and electrics.values.checkRE == 0 and electrics.values.fuel < batteryLow and electrics.values.hybridMode == "EV" then
    electrics.values.hybridMode = "HEV"
  end

  if electrics.values.ignitionLevel < 2 or glideCoef > 2.5 or engine.thermals.coolantTemperature >= (engineTempHigh + 10) or electrics.values.hybridMode ~= "HEV" then
    setMode("SERIES")
  elseif electrics.values.ignitionLevel == 2 and electrics.values.hybridMode == "HEV" then
    if electrics.values.fuel >= electrics.values.hybridSOC / 100 or electrics.values.gear ~= "D" then
      setMode("SERIES")
    elseif electrics.values.gear == "D" and electrics.values.fuel < electrics.values.hybridSOC / 100 then
      local engineCanProduceTorque = min((((engine.torqueCurve[floor(iDM_Transmission.outputAV2 * iDM_Transmission.gearRatio2 / rpmToAV)] or 0) * engine.intakeAirDensityCoef) * engine.forcedInductionCoef * electrics.values.engineTrottle + engine.nitrousOxideTorque) * engine.outputTorqueState * engine.slowIgnitionErrorCoef * engine.fastIgnitionErrorCoef, engine.maxTorqueLimit)
      if electrics.values.reRunning ~= 1 or electrics.values.brake > 0 or iDM_Transmission.outputAV2 > freeAV / iDM_Transmission.gearRatio2 then
        setMode("SERIES")
      elseif electrics.values.reRunning == 1 and electrics.values.brake == 0 and iDM_Transmission.outputAV2 <= freeAV / iDM_Transmission.gearRatio2 then
        if electrics.values.fuel >= batteryCriticalLow then
          if iDM_Transmission.outputAV2 < parallelAV / iDM_Transmission.gearRatio2 * 0.95 or (engineCanProduceTorque * iDM_Transmission.outputAV2 * iDM_Transmission.gearRatio2 * 0.975 < currentPower and enginePower < currentPower and aggressionCoef <= 0.25) then
            setMode("SERIES")
          elseif iDM_Transmission.outputAV2 > parallelAV / iDM_Transmission.gearRatio2 and (engineCanProduceTorque * iDM_Transmission.outputAV2 * iDM_Transmission.gearRatio2 >= currentPower or aggressionCoef > 0.35) then
            setMode("PARALLEL")
          end
        else
          if iDM_Transmission.outputAV2 < parallelAV / iDM_Transmission.gearRatio2 * 0.95 or aggressionCoef > 0.75 then
            setMode("SERIES")
          elseif iDM_Transmission.outputAV2 > parallelAV / iDM_Transmission.gearRatio2 and aggressionCoef <= 0.75 then
            setMode("PARALLEL")
          end
        end
      end
    end
  end

  if electrics.values.hevState == "PARALLEL" then
    engine:setTempRevLimiter(freeAV)
    startEngine()
    electrics.values.p1Throttle = 0
    electrics.values.p1RegenThrottle = 0
    engine.electricsThrottleName = "engineTrottle"
    electrics.values.engineTrottle = min(throttleSet + (electrics.values.engineThrottleOffset or 0), 1)
  else
    engine.electricsThrottleName = "engineTrottle"

    if electrics.values.ignitionLevel < 2 or electrics.values.hybridMode == "EV" or electrics.values.fuel >= electrics.values.hybridSOC / 100 or engine.thermals.coolantTemperature >= (engineTempHigh + 10) then
      idlingTime = 0
      stopEngine()
    elseif electrics.values.ignitionLevel == 2 then
      idlingTempKeeper()
      if electrics.values.gear == "P" and electrics.values.wheelspeed < 1e-2 and engine.thermals.coolantTemperature <= engineTempHigh and electrics.values.hybridMode == "RE" and input.throttle >= 0.25 then
        idlingTime = 0
        engine:setTempRevLimiter(idleAV)
        startEngine()
      elseif electrics.values.wheelspeed <= idleSpeed and electrics.values.fuel >= batteryLow and aggressionCoef < 0.75 and idlingBreak == 1 then
        engine:setTempRevLimiter(idleAV)
        idlingTime = idlingTime + dt
        if idlingTime >= 10 then
          stopEngine()
        end
      elseif electrics.values.fuel >= batteryLow and glideCoef > 2.5 and idlingBreak == 1 then
        idlingTime = idlingTime + dt
        stopEngine()
      elseif engine.thermals.coolantTemperature > engineTempHigh then
        idlingTime = 0
        engine:setTempRevLimiter(idleAV)
        if engine.thermals.coolantTemperature < (engineTempHigh + 5) then
          startEngine()
        end
      elseif electrics.values.fuel >= batteryCriticalLow then
        idlingTime = 0
        local minRPM = clamp(currentPowerCoef * 100, idleRPM, freeRPM)
        local maxRPM = clamp(sqrt(electrics.values.wheelspeed) * 800, minRPM, freeRPM)
        engine:setTempRevLimiter((minRPM + (maxRPM - minRPM) * aggressionCoef) * rpmToAV)
        startEngine()
      else
        idlingTime = 0
        local minRPM = clamp(currentPowerCoef * 120, idleRPM, freeRPM)
        local maxRPM = clamp(sqrt(electrics.values.wheelspeed) * 1200, idleRPM, freeRPM)
        engine:setTempRevLimiter((minRPM + (maxRPM - minRPM) * aggressionCoef) * rpmToAV)
        startEngine()
      end
    end

    if electrics.values.reRunning == 1 then
      electrics.values.engineTrottle = throttleSet
      if engine.outputAV1 < engine.tempRevLimiterAV * 0.9 then
        electrics.values.p1RegenThrottle = max(0, electrics.values.p1RegenThrottle - 0.02)
      end
      if engine.throttle < throttleSet then
        electrics.values.p1RegenThrottle = min(1, electrics.values.p1RegenThrottle + 0.02)
      end
    else
      electrics.values.engineTrottle = 0
    end
  end
end

local function updateGaugeInfo(dt)
  local totalCapacity = 0
  local remainingEnergy = 0
  local remainingVolume = 0
  
  for _, s in pairs(engine.registeredEnergyStorages) do
    local storage = energyStorage.getStorage(s)
    if storage and storage.type ~= "n2oTank" then
      totalCapacity = totalCapacity + storage.energyCapacity
      remainingEnergy = remainingEnergy + storage.storedEnergy
      remainingVolume = remainingVolume + storage.remainingVolume
    end
  end
  local remainingGasolineRatio = remainingEnergy / totalCapacity
  electrics.values.remainingGasolineRatio = remainingGasolineRatio
  if remainingGasolineRatio <= 0.1 then
    electrics.values.lowGasoline = 1
  else
    electrics.values.lowGasoline = 0
  end

  local motorThrottle = electrics.values.throttle
  local motorRegenThrottle = electrics.values.regenThrottle

  if electrics.values.hevState == "PARALLEL" then
    if electrics.values.throttle > 0 then
      electrics.values.hybridEnergyFlow = "P-"
    elseif electrics.values.throttle <= 0 and electrics.values.regenThrottle == 0 then
      electrics.values.hybridEnergyFlow = "PP"
    elseif electrics.values.throttle <= 0 and electrics.values.regenThrottle > 0 then
      if unsmoothedCurrentPower <= 0 then
        electrics.values.hybridEnergyFlow = "PR"
      else
        electrics.values.hybridEnergyFlow = "P+"
      end
    end
  elseif electrics.values.hevState == "SERIES" then
    if electrics.values.reRunning == 1 and electrics.values.wheelspeed >= 0.1 then
      if motorThrottle > 0 then
        if electrics.values.fuel < lastBatteryLevel then
          electrics.values.hybridEnergyFlow = "S-"
        elseif electrics.values.fuel == lastBatteryLevel then
          electrics.values.hybridEnergyFlow = "SS"
        elseif electrics.values.fuel > lastBatteryLevel then
          electrics.values.hybridEnergyFlow = "S+"
        end
      elseif motorThrottle <= 0 and motorRegenThrottle == 0 then
        electrics.values.hybridEnergyFlow = "SC"
      elseif motorThrottle <= 0 and motorRegenThrottle > 0 then
        electrics.values.hybridEnergyFlow = "SR"
      end
    elseif electrics.values.reRunning == 0 and electrics.values.wheelspeed >= 0.1 then
      if motorThrottle > 0 then
        electrics.values.hybridEnergyFlow = "E-"
      elseif motorThrottle <= 0 and motorRegenThrottle == 0 then
        electrics.values.hybridEnergyFlow = "EE"
      elseif motorThrottle <= 0 and motorRegenThrottle > 0 then
        electrics.values.hybridEnergyFlow = "ER"
      end
    elseif electrics.values.wheelspeed < 0.1 then
      if electrics.values.reRunning == 1 then
        electrics.values.hybridEnergyFlow = "SC"
      else
        electrics.values.hybridEnergyFlow = "EE"
      end
    end
  end
  lastBatteryLevel = electrics.values.fuel
end

local function updateGFX(dt)
  updateMotorIgnition()
  updateEHSState(dt)
  updateGaugeInfo(dt)
end

local function serialize()
end

local function deserialize(data)
end

local function init(jbeamData)
  extensions.load("socControl")
  engine = powertrain.getDevice(jbeamData.engineName) or powertrain.getDevice("mainEngine")
  iDM_Transmission = powertrain.getDevice(jbeamData.iDM_Transmission_Name) or powertrain.getDevice(jbeamData.p1Name)
  p1_Motors = powertrain.getDevicesByType("p1_Motor")
  p3_Motors = powertrain.getDevicesByType("p3_Motor")
  motors = powertrain.getDevicesByType("electricMotor")
  batteryLow = jbeamData.batteryLow or 0.15
  batteryCriticalLow = jbeamData.batteryCriticalLow or 0.05
  engineTempReady = jbeamData.engineTempReady or 85
  engineTempHigh = jbeamData.engineTempHigh or 100
  engineTempLow = jbeamData.engineTempLow or (engineTempReady - 5)
  freeRPM = jbeamData.freeRPM or 5500
  idleRPM = jbeamData.idleRPM or 1500
  parallelRPM = jbeamData.parallelRPM or 1200
  freeAV = freeRPM * rpmToAV
  idleAV = idleRPM * rpmToAV
  parallelAV = parallelRPM * rpmToAV
  idleSpeed = (jbeamData.idleSpeed or 15) * kphToMs

  electrics.values.checkRE = engine.isDisabled and 1 or 0

  local minV = nil
  for k, v in pairs(engine.invBurnEfficiencyTable) do
    if minV == nil or v < minV or (v == minV and k < throttleSet) then
      minV = v
      throttleSet = k
    end
  end
  throttleSet = throttleSet / 100
end

local function initLastStage(jbeamData)
end

M.init = init
M.initLastStage = initLastStage
M.updateMotorIgnition = updateMotorIgnition
M.updateEHSState = updateEHSState
M.updateGFX = updateGFX
M.serialize = serialize
M.deserialize = deserialize

return M
