-- written by DaddelZeit
-- DO NOT USE WITHOUT PERMISSION

local M = {}
M.type = "auxiliary"
M.defaultOrder = 80

M.isActive = false
M.isActing = false

local CMU = nil
local isDebugEnabled = false

local controlParameters = {isEnabled = true}
local initialControlParameters

local configPacket = {sourceType = "adaptiveAirSuspension", packetType = "config", config = controlParameters}
local debugPacket = {sourceType = "adaptiveAirSuspension"}

local lastRunning
local beamModes = {}
local currentMode = nil
local airBeamsFront = {}
local airBeamsRear = {}

local currentPressureFront = 120
local currentPressureRear = 70
local initialHeightFront = 120
local initialHeightRear = 70
local pressureLimits = {
  frontHigh = 120,
  rearHigh = 90,
  frontLow = 74,
  rearLow = 1
}

local psiToPascal = 6894.757293178

local function setPressureDirect(frontBeamPressurePSI, rearBeamPressurePSI)
  for _, cid in ipairs(airBeamsFront) do
    local beam = v.data.beams[cid]
    local maxBeamPressure = beam.maxPressure
    local spring = beam.beamSpring
    local damp = beam.beamDamp
    obj:setBeamPressureRel(cid, frontBeamPressurePSI * psiToPascal, maxBeamPressure, spring, damp)
  end
  for _, cid in ipairs(airBeamsRear) do
    local beam = v.data.beams[cid]
    local maxBeamPressure = beam.maxPressure
    local spring = beam.beamSpring
    local damp = beam.beamDamp
    obj:setBeamPressureRel(cid, rearBeamPressurePSI * psiToPascal, maxBeamPressure, spring, damp)
  end
end

local function setMode(modeName)
  local mode = beamModes[modeName]
  if not mode then
    log("E", "adaptiveAirSuspension.setMode", "Can't find mode: " .. modeName)
    return
  end
  currentMode = mode

  for _, cid in ipairs(airBeamsFront) do
    local beam = v.data.beams[cid]
    local beamPressurePSI = beam.beamPressurePSI * mode.beamPressurePSICoefFront
    local maxBeamPressure = beam.maxPressure
    local spring = beam.spring
    local damp = beam.damp
    obj:setBeamPressureRel(cid, beamPressurePSI, maxBeamPressure, spring, damp)
    currentPressureFront = beam.beamPressurePSI
  end
  for _, cid in ipairs(airBeamsRear) do
    local beam = v.data.beams[cid]
    local beamPressurePSI = beam.beamPressurePSI * mode.beamPressurePSICoefRear
    local maxBeamPressure = beam.maxPressure
    local spring = beam.spring
    local damp = beam.damp
    obj:setBeamPressureRel(cid, beamPressurePSI * psiToPascal, maxBeamPressure, spring, damp)
    currentPressureRear = beam.beamPressurePSI
  end
end

local function reset()
  currentPressureFront = 120
  currentPressureRear = 70
  lastRunning = electrics.values.running
end

local bounceTimer = 0
local bounceEnabled = 0
local pressureFrontSmoother = newExponentialSmoothingT(40)
local pressureRearSmoother = newExponentialSmoothingT(40)

local previousPressureFront, previousPressureRear = 120, 70
local function toggleBounceMode()
  if bounceEnabled == 0 then
    previousPressureFront, previousPressureRear = currentPressureFront, currentPressureRear
  end
  bounceEnabled = (1-bounceEnabled) or 0
  electrics.values.airSuspensionBounceMode = 0

  currentPressureFront, currentPressureRear = previousPressureFront, previousPressureRear
end

local engineOff = false
local function updateGFX(dt)
  if lastRunning == 1 and electrics.values.running == 0 then
    -- car just turned off
    previousPressureFront, previousPressureRear = currentPressureFront, currentPressureRear
    engineOff = true
  elseif lastRunning == 0 and electrics.values.running == 1 then
    -- car just turned on
    currentPressureFront, currentPressureRear = previousPressureFront, previousPressureRear
    engineOff = false
  end

  if not engineOff then
    if bounceEnabled == 1 then
      bounceTimer = bounceTimer + dt + 0.004
      if bounceTimer < 0.4 then
        currentPressureFront = 44
        currentPressureRear = -40
      elseif bounceTimer > 0.4 and bounceTimer < 0.8 then
        currentPressureFront = 147
        currentPressureRear = 140
      elseif bounceTimer > 0.8 then
        bounceTimer = 0
      end
      electrics.values.airSuspensionBounceMode = 1
      if electrics.values.wheelspeed*3.6 > 30 then toggleBounceMode() end
    end
  else
    -- car is off, lower it to min
    currentPressureFront = pressureLimits.frontLow
    currentPressureRear = pressureLimits.rearLow
  end

  setPressureDirect(pressureFrontSmoother:get(currentPressureFront), pressureRearSmoother:get(currentPressureRear))
  lastRunning = electrics.values.running
end

local function changePSI(val)
  currentPressureFront = clamp(currentPressureFront + val, pressureLimits.frontLow, pressureLimits.frontHigh)
  currentPressureRear = clamp(currentPressureRear + val, pressureLimits.rearLow, pressureLimits.rearHigh)

  if currentPressureFront == pressureLimits.frontHigh then
    electrics.values.airSuspensionLimitHigh = 1
  else electrics.values.airSuspensionLimitHigh = 0 end

  if currentPressureFront == pressureLimits.frontLow then
    electrics.values.airSuspensionLimitLow = 1
  else electrics.values.airSuspensionLimitLow = 0 end
end

local function resetHeight()
  currentPressureFront = initialHeightFront
  currentPressureRear = initialHeightRear

  electrics.values.airSuspensionLimitHigh = 0
  electrics.values.airSuspensionLimitLow = 0
end

local function init(jbeamData)
  local airBeamNamesFront = jbeamData.airBeamNamesFront or {}
  local airBeamNamesRear = jbeamData.airBeamNamesRear or {}

  pressureLimits = jbeamData.pressureLimits or pressureLimits

  lastRunning = electrics.values.running
  currentPressureFront = 120
  currentPressureRear = 70

  electrics.values.airSuspensionLimitHigh = 0
  electrics.values.airSuspensionLimitLow = 0

  airBeamsFront = {}
  airBeamsRear = {}
  for _, b in pairs(v.data.beams) do
    if b.name then
      for _, name in pairs(airBeamNamesFront) do
        if b.name == name then
          table.insert(airBeamsFront, b.cid)
        end
      end
      for _, name in pairs(airBeamNamesRear) do
        if b.name == name then
          table.insert(airBeamsRear, b.cid)
        end
      end
    end
  end

  local modeData = tableFromHeaderTable(jbeamData.modes or {})

  beamModes = {}
  for _, mode in pairs(modeData) do
    beamModes[mode.name] = {
      beamPressurePSICoefFront = mode.beamPressurePSICoefFront or 1,
      beamPressurePSICoefRear = mode.beamPressurePSICoefRear or 1,
    }
  end

  local nameString = jbeamData.name
  local slashPos = nameString:find("/", -nameString:len())
  if slashPos then
    nameString = nameString:sub(slashPos + 1)
  end
  debugPacket.sourceName = nameString

  M.isActive = true
end

local function initLastStage()
end

local function setDebugMode(debugEnabled)
  isDebugEnabled = debugEnabled
end

local function registerCMU(cmu)
  CMU = cmu
end

local function shutdown()
  M.isActive = false
  M.updateGFX = nil
  M.update = nil
end

local function setParameters(parameters)
  if parameters.damperMode then
    setMode(parameters.damperMode)
  end
end

local function setConfig(configTable)
  controlParameters = configTable
end

local function getConfig()
  return deepcopy(controlParameters)
end

local function sendConfigData()
  configPacket.config = controlParameters
  CMU.sendDebugPacket(configPacket)
end

M.init = init
M.reset = reset
M.initLastStage = initLastStage
M.changePSI = changePSI
M.resetHeight = resetHeight

M.registerCMU = registerCMU
M.setDebugMode = setDebugMode
M.shutdown = shutdown
M.setParameters = setParameters
M.setConfig = setConfig
M.getConfig = getConfig
M.sendConfigData = sendConfigData

-- funsies
M.updateGFX = updateGFX
M.toggleBounceMode = toggleBounceMode

return M
